'use client'

import { useEffect, useState, useRef } from 'react'
import { useParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Pago, Cliente, Suscripcion } from '@/lib/types'
import { format } from 'date-fns'
import Link from 'next/link'
import Image from 'next/image'
import jsPDF from 'jspdf'
import html2canvas from 'html2canvas'

export default function FacturaPage() {
  const params = useParams()
  const [pago, setPago] = useState<Pago | null>(null)
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [suscripcion, setSuscripcion] = useState<Suscripcion | null>(null)
  const [loading, setLoading] = useState(true)
  const [generandoPDF, setGenerandoPDF] = useState(false)
  const [numeroFactura, setNumeroFactura] = useState<number>(0)
  const facturaRef = useRef<HTMLDivElement>(null)

  // RUT de la empresa emisora
  const RUT_EMPRESA = '77.317.557-8'
  const DIRECCION_EMPRESA = 'Av Providencia 2296 - Santiago - Metropolitana - Chile'
  const TELEFONO_EMPRESA = '+56979787013'

  useEffect(() => {
    if (params.id) {
      fetchPago(params.id as string)
      generarNumeroFactura()
    }
  }, [params.id])

  async function generarNumeroFactura() {
    try {
      // Obtener el último número de factura del año actual
      const año = new Date().getFullYear()
      const { count } = await supabase
        .from('pagos')
        .select('*', { count: 'exact', head: true })
        .gte('created_at', `${año}-01-01`)
        .lt('created_at', `${año + 1}-01-01`)

      // El número de factura será el count + 1140 (para comenzar desde 1140)
      const numeroBase = 1140
      setNumeroFactura((count || 0) + numeroBase)
    } catch (error) {
      console.error('Error generando número de factura:', error)
      // Si hay error, usar el número base mínimo
      setNumeroFactura(1140)
    }
  }

  async function fetchPago(id: string) {
    try {
      const { data, error } = await supabase
        .from('pagos')
        .select(`
          *,
          cliente:clientes(*),
          suscripcion:suscripciones(*)
        `)
        .eq('id', id)
        .single()

      if (error) throw error
      setPago(data)
      setCliente(data.cliente as Cliente)
      setSuscripcion(data.suscripcion as Suscripcion | null)
    } catch (error) {
      console.error('Error fetching pago:', error)
    } finally {
      setLoading(false)
    }
  }

  async function descargarPDF() {
    if (!facturaRef.current || !pago || !cliente) return

    setGenerandoPDF(true)

    try {
      const canvas = await html2canvas(facturaRef.current, {
        scale: 2,
        useCORS: true,
        logging: false,
      })

      const imgData = canvas.toDataURL('image/png')
      const pdf = new jsPDF('p', 'mm', 'a4')
      const pdfWidth = pdf.internal.pageSize.getWidth()
      const pdfHeight = pdf.internal.pageSize.getHeight()
      const imgWidth = canvas.width
      const imgHeight = canvas.height
      const ratio = Math.min(pdfWidth / imgWidth, pdfHeight / imgHeight)
      const imgX = (pdfWidth - imgWidth * ratio) / 2
      const imgY = 0

      pdf.addImage(imgData, 'PNG', imgX, imgY, imgWidth * ratio, imgHeight * ratio)
      pdf.save(`factura-${numeroFactura}.pdf`)
    } catch (error) {
      console.error('Error generating PDF:', error)
      alert('Error al generar el PDF')
    } finally {
      setGenerandoPDF(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando factura...</div>
  }

  if (!pago || !cliente) {
    return (
      <div className="text-center p-8">
        <h1 className="text-2xl font-bold mb-4">Factura no encontrada</h1>
        <Link href="/pagos" className="text-blue-600 hover:underline">Volver a pagos</Link>
      </div>
    )
  }

  // Función para formatear números en formato chileno (1.000 o 1.000,50)
  function formatearNumero(numero: number): string {
    // Si el número es entero, mostrar sin decimales
    if (numero % 1 === 0) {
      return numero.toLocaleString('es-CL', {
        minimumFractionDigits: 0,
        maximumFractionDigits: 0
      })
    }
    // Si tiene decimales, mostrar con 2 decimales
    return numero.toLocaleString('es-CL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    })
  }

  // Calcular montos según si incluye IVA o no
  let netAmount: number
  let iva: number
  let total: number

  if (pago.incluir_iva) {
    // Si incluye IVA, el monto es el NETO y el IVA se ADICIONA
    netAmount = pago.monto // El monto es el neto
    iva = netAmount * 0.19 // IVA (19% del neto)
    total = netAmount + iva // Total = Neto + IVA
  } else {
    // Si no incluye IVA, el monto es el total
    netAmount = pago.monto
    iva = 0
    total = pago.monto
  }

  // Usar fecha_inicio del pago si existe, sino usar fecha_pago
  // Trabajar con strings YYYY-MM-DD para evitar problemas de zona horaria
  const fechaBaseStr = pago.fecha_inicio || pago.fecha_pago
  const fechaBase = fechaBaseStr.split('T')[0] // Asegurar formato YYYY-MM-DD
  
  // Fecha de vencimiento: usar fecha_fin del pago si existe, sino calcular 5 días después
  let fechaVencimientoStr: string
  if (pago.fecha_fin) {
    fechaVencimientoStr = pago.fecha_fin.split('T')[0]
  } else {
    // Calcular 5 días después de la fecha base
    const [year, month, day] = fechaBase.split('-').map(Number)
    const fechaVenc = new Date(year, month - 1, day)
    fechaVenc.setDate(fechaVenc.getDate() + 5)
    const yearVenc = fechaVenc.getFullYear()
    const monthVenc = (fechaVenc.getMonth() + 1).toString().padStart(2, '0')
    const dayVenc = fechaVenc.getDate().toString().padStart(2, '0')
    fechaVencimientoStr = `${yearVenc}-${monthVenc}-${dayVenc}`
  }
  
  // Formatear fechas para mostrar (sin usar Date para evitar problemas de zona horaria)
  function formatearFecha(fechaStr: string): string {
    const [year, month, day] = fechaStr.split('-').map(Number)
    return `${day.toString().padStart(2, '0')}/${month.toString().padStart(2, '0')}/${year}`
  }
  
  const fechaMostrar = formatearFecha(fechaBase)
  const fechaVencimientoMostrar = formatearFecha(fechaVencimientoStr)

  return (
    <div className="min-h-screen bg-gray-100 py-8">
      <div className="max-w-4xl mx-auto px-4">
        <div className="bg-white p-6 rounded-lg shadow mb-4">
          <div className="flex justify-between items-center">
            <Link href="/pagos" className="text-blue-600 hover:underline">← Volver a pagos</Link>
            <button
              onClick={descargarPDF}
              disabled={generandoPDF}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50 transition"
            >
              {generandoPDF ? 'Generando...' : '📄 Descargar PDF'}
            </button>
          </div>
        </div>

        <div ref={facturaRef} className="bg-white border-2 border-gray-300 p-8 shadow-lg" style={{ minHeight: '400px' }}>
          {/* Header - Logo y datos de la empresa */}
          <div className="flex justify-between items-start mb-6">
            <div>
              <div className="mb-3">
                <Image
                  src="/img/logo_navigps.png"
                  alt="NAVIGPS Logo"
                  width={150}
                  height={75}
                  className="object-contain"
                />
              </div>
              <div>
                <p className="text-sm text-gray-700">{DIRECCION_EMPRESA}</p>
                <p className="text-sm text-gray-700">Tel: {TELEFONO_EMPRESA}</p>
                <p className="text-sm text-gray-700">
                  Web: <a href="https://navigps.cl" className="text-blue-600 hover:underline">navigps.cl</a>
                </p>
                <p className="text-sm text-gray-700">
                  <a href="mailto:info@navigps.cl" className="text-blue-600 hover:underline">info@navigps.cl</a>
                </p>
              </div>
            </div>

            {/* Bloque de identificación de factura */}
            <div className="border-2 border-black p-4 text-right min-w-[200px]">
              <p className="font-semibold mb-1">RUT: {RUT_EMPRESA}</p>
              <p className="font-bold text-lg mb-2">FACTURA ELECTRÓNICA</p>
              <p className="font-semibold">Nº: {numeroFactura}</p>
            </div>
          </div>

          {/* Datos del cliente */}
          <div className="grid grid-cols-2 gap-4 mb-6 text-sm">
            <div>
              <p><span className="font-semibold">Fecha:</span> {fechaMostrar}</p>
            </div>
            <div>
              <p><span className="font-semibold">Fecha de vencimiento:</span> {fechaVencimientoMostrar}</p>
            </div>
            <div>
              <p><span className="font-semibold">Cliente:</span> {cliente.nombre || cliente.empresa || 'Cliente sin nombre'}</p>
            </div>
            {cliente.empresa && (
              <div>
                <p><span className="font-semibold">Empresa:</span> {cliente.empresa}</p>
              </div>
            )}
            {cliente.rfc && (
              <div>
                <p><span className="font-semibold">RUT:</span> {cliente.rfc}</p>
              </div>
            )}
            {cliente.direccion && (
              <div className="col-span-2">
                <p><span className="font-semibold">Dirección:</span> {cliente.direccion}</p>
              </div>
            )}
            <div>
              <p><span className="font-semibold">Condiciones de pago:</span></p>
            </div>
            <div>
              <p><span className="font-semibold">GIRO:</span></p>
            </div>
          </div>

          {/* Tabla de items */}
          <div className="mb-6">
            <table className="w-full border-collapse border border-gray-300">
              <thead>
                <tr className="bg-gray-100">
                  <th className="border border-gray-300 px-4 py-2 text-left font-semibold">DESCRIPCIÓN</th>
                  <th className="border border-gray-300 px-4 py-2 text-center font-semibold">CANTIDAD</th>
                  <th className="border border-gray-300 px-4 py-2 text-right font-semibold">PRECIO UNITARIO</th>
                  <th className="border border-gray-300 px-4 py-2 text-right font-semibold">IMPORTE</th>
                </tr>
              </thead>
              <tbody>
                <tr>
                  <td className="border border-gray-300 px-4 py-3">
                    {suscripcion ? (
                      <>
                        [{suscripcion.tipo === 'mensual' ? 'M10' : 'A10'}] PLAN GPS {suscripcion.tipo === 'mensual' ? 'FULL M10' : 'FULL A10'}. 
                        Acceso continuo a nuestros servicios GPS durante un ciclo de facturación de {suscripcion.tipo === 'mensual' ? 'un mes' : 'un año'}.
                      </>
                    ) : (
                      `Servicio GPS - ${pago.notas || 'Pago de servicios'}`
                    )}
                  </td>
                  <td className="border border-gray-300 px-4 py-3 text-center">
                    {suscripcion?.cantidad || 1}
                  </td>
                  <td className="border border-gray-300 px-4 py-3 text-right">
                    ${formatearNumero(suscripcion?.cantidad ? netAmount / suscripcion.cantidad : netAmount)}
                  </td>
                  <td className="border border-gray-300 px-4 py-3 text-right font-semibold">${formatearNumero(netAmount)}</td>
                </tr>
              </tbody>
            </table>
          </div>

          {/* Resumen de montos */}
          <div className="flex justify-end mb-6">
            <table className="w-64 border-collapse">
              <tbody>
                <tr>
                  <td className="px-4 py-2 text-right border border-gray-300">Net Amount</td>
                  <td className="px-4 py-2 text-right border border-gray-300 font-semibold">${formatearNumero(netAmount)}</td>
                </tr>
                {pago.incluir_iva && iva > 0 && (
                  <tr>
                    <td className="px-4 py-2 text-right border border-gray-300">VAT 19.00</td>
                    <td className="px-4 py-2 text-right border border-gray-300 font-semibold">${formatearNumero(iva)}</td>
                  </tr>
                )}
                <tr className="bg-blue-50">
                  <td className="px-4 py-2 text-right border border-gray-300 font-bold">Total</td>
                  <td className="px-4 py-2 text-right border border-gray-300 font-bold text-blue-600">${formatearNumero(total)}</td>
                </tr>
              </tbody>
            </table>
          </div>

          {/* Footer */}
          <div className="border-t border-gray-300 pt-4 text-center text-sm text-gray-600">
            <p>info@navigps.cl https://navigps.cl</p>
          </div>
        </div>
      </div>
    </div>
  )
}
